const videoElement = document.getElementById("video");
const canvasElement = document.getElementById("output");
const ctx = canvasElement.getContext("2d");

function shuffle(array) {
  for (let i = array.length - 1; i > 0; i--) {
    const j = Math.floor(Math.random() * (i + 1));
    [array[i], array[j]] = [array[j], array[i]];
  }
  return array;
}

const bgImg = new Image();
bgImg.src = 'bg.jpg';

const helmetSources = ['assets/1.png', 'assets/2.png', 'assets/3.png', 'assets/4.png', 'assets/5.png'];
let helmetImgs = [];

function resetHelmetImgs() {
  const shuffledSources = shuffle([...helmetSources]);
  helmetImgs = shuffledSources.map(src => {
    const img = new Image();
    img.src = src;
    return img;
  });
}

resetHelmetImgs();

const landscapeBorder = new Image();
landscapeBorder.src = 'assets/landscape.png';
const portraitBorder = new Image();
portraitBorder.src = 'assets/potrait.png';

async function initCamera() {
  const stream = await navigator.mediaDevices.getUserMedia({ video: true });
  videoElement.srcObject = stream;
  videoElement.addEventListener('loadedmetadata', () => {
    canvasElement.width = 1280;
    canvasElement.height = 1080;
  });
}

const faceMesh = new FaceMesh({
  locateFile: (file) => `https://cdn.jsdelivr.net/npm/@mediapipe/face_mesh/${file}`,
});
faceMesh.setOptions({
  maxNumFaces: 5,
  refineLandmarks: true,
  minDetectionConfidence: 0.5,
  minTrackingConfidence: 0.5,
});

const hands = new Hands({
  locateFile: (file) => `https://cdn.jsdelivr.net/npm/@mediapipe/hands/${file}`,
});
hands.setOptions({
  maxNumHands: 2,
  modelComplexity: 1,
  minDetectionConfidence: 0.5,
  minTrackingConfidence: 0.5,
});

let snapped = false;
let countingDown = false;
let cooldown = false;
let latestFaceResults = null;
let uploadedFilename = null;
const qrContainer = document.getElementById("qr-container");
const qrCodeImg = document.getElementById("qr-code");
const countdownEl = document.getElementById("countdown");
let countdownInterval = null;

function isPeaceSign(handLandmarks) {
  const indexTip = handLandmarks[8];
  const indexPIP = handLandmarks[6];
  const middleTip = handLandmarks[12];
  const middlePIP = handLandmarks[10];
  const ringTip = handLandmarks[16];
  const ringPIP = handLandmarks[14];
  const pinkyTip = handLandmarks[20];
  const pinkyPIP = handLandmarks[18];
  const thumbTip = handLandmarks[4];
  const thumbIP = handLandmarks[3];

  const indexExtended = indexTip.y < indexPIP.y;
  const middleExtended = middleTip.y < middlePIP.y;
  const ringExtended = ringTip.y < ringPIP.y;
  const pinkyExtended = pinkyTip.y < pinkyPIP.y;
  const thumbExtended = thumbTip.y < thumbIP.y;

  return (
    indexExtended &&
    middleExtended &&
    !ringExtended &&
    !pinkyExtended &&
    !thumbExtended
  );
}

hands.onResults((results) => {
  if (results.multiHandLandmarks && results.multiHandLandmarks.length > 0) {
    const peaceDetected = results.multiHandLandmarks.some((hand) =>
      isPeaceSign(hand)
    );
    if (peaceDetected && !snapped && !countingDown && !cooldown) {
      startCountdown();
    }
  }
});

function startCountdown() {
  console.log("Peace sign detected, starting countdown before snapping photo.");
  countingDown = true;
  cooldown = true;
  const snapCountdownContainer = document.getElementById("snap-countdown-container");
  const snapCountdownEl = document.getElementById("snap-countdown");
  snapCountdownContainer.style.display = "block";
  document.getElementById('instruction-overlay').style.display = 'none';
  let snapCount = 5;
  snapCountdownEl.textContent = snapCount;
  const snapInterval = setInterval(() => {
    snapCount--;
    if (snapCount > 0) {
      snapCountdownEl.textContent = snapCount;
    } else {
      clearInterval(snapInterval);
      snapCountdownContainer.style.display = "none";
      countingDown = false;
      console.log("Countdown finished, snapping photo now.");
      snapPhoto();
    }
  }, 1000);
}

function snapPhoto() {
  if (canvasElement.width === 0 || canvasElement.height === 0) {
    console.error("Canvas not ready for snapping.");
    snapped = false;
    cooldown = false;
    return;
  }
  snapped = true;
  console.log("Snapping photo now.");
  // White flash
  const flashEl = document.getElementById("flash");
  flashEl.style.display = "block";
  setTimeout(() => {
    flashEl.style.display = "none";
  }, 200);
  try {
    if (!latestFaceResults || !latestFaceResults.image) {
      console.error("No face results available for snapping.");
      snapped = false;
      cooldown = false;
      return;
    }
    const tempCanvas = document.createElement('canvas');
    tempCanvas.width = 1280;
    tempCanvas.height = 1080;
    const tempCtx = tempCanvas.getContext('2d');
    tempCtx.save();
    tempCtx.scale(-1, 1);
    tempCtx.translate(-tempCanvas.width, 0);
    tempCtx.drawImage(videoElement, 0, 0, tempCanvas.width, tempCanvas.height);
    tempCtx.restore();
    if (latestFaceResults.multiFaceLandmarks && latestFaceResults.multiFaceLandmarks.length > 0) {
      latestFaceResults.multiFaceLandmarks.sort((a, b) => a[1].x - b[1].x);
      latestFaceResults.multiFaceLandmarks.forEach((face, index) => {
        const xs = face.map((pt) => pt.x);
        const ys = face.map((pt) => pt.y);
        const minX = Math.min(...xs) * tempCanvas.width;
        const maxX = Math.max(...xs) * tempCanvas.width;
        const minY = Math.min(...ys) * tempCanvas.height;
        const maxY = Math.max(...ys) * tempCanvas.height;
        const faceWidth = maxX - minX;
        const faceHeight = maxY - minY;

        // Use eye landmarks to position helmet more precisely
        // Left eye: landmark 33, Right eye: landmark 263 (MediaPipe Face Mesh)
        const leftEye = face[33];
        const rightEye = face[263];
        const eyeCenterX = ((leftEye.x + rightEye.x) / 2) * tempCanvas.width;
        const eyeCenterY = ((leftEye.y + rightEye.y) / 2) * tempCanvas.height;

        // Calculate rotation angle based on eye line
        const angle = Math.atan2(rightEye.y - leftEye.y, rightEye.x - leftEye.x);

        const helmetSize = faceWidth * 2.0;

        // Mirror the eye center X for correct positioning in mirrored view
        const mirroredEyeCenterX = tempCanvas.width - eyeCenterX;

        // Assign helmet index based on face index for consistency
        let helmetIndex = index % helmetImgs.length;

        if (helmetImgs[helmetIndex] && helmetImgs[helmetIndex].complete) {
          tempCtx.save();
          tempCtx.translate(mirroredEyeCenterX, eyeCenterY);
          tempCtx.rotate(-angle); // Negate angle for mirrored view
          tempCtx.drawImage(
            helmetImgs[helmetIndex],
            -helmetSize / 2,
            -helmetSize / 2.0,
            helmetSize,
            helmetSize
          );
          tempCtx.restore();
        }
      });
    }
    // Add border
    const borderImg = tempCanvas.width > tempCanvas.height ? landscapeBorder : portraitBorder;
    if (borderImg.complete) {
      tempCtx.drawImage(borderImg, 0, 0, tempCanvas.width, tempCanvas.height);
    }
    // Set the snapped photo for display
    document.getElementById('snapped-photo').src = tempCanvas.toDataURL('image/jpeg');
    const filename = 'photo_' + Date.now() + '.jpg';
    const loadingEl = document.getElementById("loading");
    const qrCodeImg = document.getElementById("qr-code");
    canvasElement.style.display = "none";
    qrContainer.style.display = "block";
    loadingEl.style.display = "block";
    // Upload to server
    tempCanvas.toBlob((blob) => {
      console.log('Blob size:', blob ? blob.size : 'null');
      const formData = new FormData();
      formData.append('file', blob, filename);
          fetch('./upload.php', {
        method: 'POST',
        body: formData
      }).then(response => {
        console.log('Upload response status:', response.status);
        return response.text().then(text => {
          console.log('Upload response text:', text);
          try {
            return JSON.parse(text);
          } catch (e) {
            throw new Error('Invalid JSON response: ' + text);
          }
        });
      }).then(data => {
        if (data.status === 'success') {
    const imageUrl = window.location.origin + data.url;
    uploadedFilename = data.url.split('/z').pop();
    // Generate QR code
              QRCode.toDataURL(imageUrl, { width: 256, height: 256 }, (err, url) => {
                if (err) {
                  console.error("Error generating QR code:", err);
                  return;
                }
                qrCodeImg.src = url;
                loadingEl.style.display = "none";
                document.getElementById('snapped-photo').style.display = "block";
                qrCodeImg.style.display = "block";
                countdownEl.style.display = "block";
                let count = 20;
                countdownEl.textContent = count;
                if (countdownInterval) {
                  clearInterval(countdownInterval);
                }
                countdownInterval = setInterval(() => {
                  count--;
                  countdownEl.textContent = count;
                  if (count <= 0) {
                    clearInterval(countdownInterval);
                    countdownInterval = null;
                    qrContainer.style.display = "none";
                    canvasElement.style.display = "block";
                    snapped = false;
                    countdownEl.textContent = "20";
                    document.getElementById('snapped-photo').style.display = "none";
                    qrCodeImg.style.display = "none";
                    countdownEl.style.display = "none";
                    console.log("Countdown finished, returning to camera view.");
                    uploadedFilename = null;
  cooldown = false;
  resetHelmetImgs();
  document.getElementById('instruction-overlay').style.display = 'flex';
  document.getElementById('snap-countdown-container').style.display = 'none';
                  }
                }, 1000);
              });
            } else {
              console.error("Upload failed:", data.message);
              loadingEl.textContent = "Upload failed: " + data.message;
            }
          }).catch(error => {
            console.error("Error uploading:", error);
            loadingEl.textContent = "Upload failed";
          });
        });
        // Safeguard timeout to reset snapped flag in case upload fails
        setTimeout(() => {
          if (snapped && !countdownInterval) {
            console.warn("Safeguard: resetting snapped flag due to timeout.");
            snapped = false;
            countingDown = false;
            cooldown = false;
            qrContainer.style.display = "none";
            document.getElementById('snapped-photo').style.display = "none";
            canvasElement.style.display = "block";
            countdownEl.textContent = "20";
            qrCodeImg.style.display = "none";
            countdownEl.style.display = "none";
            loadingEl.style.display = "none";
          }
        }, 30000);
      } catch (error) {
        console.error("Error during photo capture:", error);
        snapped = false;
        countingDown = false;
        cooldown = false;
        qrContainer.style.display = "none";
        canvasElement.style.display = "block";
        countdownEl.textContent = "20";
      }
    }

    const resetBtn = document.getElementById("reset-btn");
    resetBtn.addEventListener("click", () => {
      console.log("Manual reset triggered.");
      snapped = false;
      countingDown = false;
      qrContainer.style.display = "none";
      document.getElementById('snapped-photo').style.display = "none";
      canvasElement.style.display = "block";
      countdownEl.textContent = "10";
      resetBtn.style.display = "none";
      document.getElementById('instruction-overlay').style.display = 'flex';
      document.getElementById('snap-countdown-container').style.display = 'none';
    });

    function showResetButton() {
      resetBtn.style.display = "inline-block";
    }

    function hideResetButton() {
      resetBtn.style.display = "none";
    }

    faceMesh.onResults((results) => {
      latestFaceResults = results;
      if (!snapped && canvasElement.width > 0) {
        ctx.clearRect(0, 0, canvasElement.width, canvasElement.height);
        ctx.save();
        ctx.scale(-1, 1);
        ctx.translate(-canvasElement.width, 0);
        ctx.drawImage(results.image, 0, 0, canvasElement.width, canvasElement.height);
        ctx.restore();

        if (results.multiFaceLandmarks && results.multiFaceLandmarks.length > 0) {
          results.multiFaceLandmarks.sort((a, b) => a[1].x - b[1].x);
          results.multiFaceLandmarks.forEach((face, index) => {
            const xs = face.map((pt) => pt.x);
            const ys = face.map((pt) => pt.y);
            const minX = Math.min(...xs) * canvasElement.width;
            const maxX = Math.max(...xs) * canvasElement.width;
            const minY = Math.min(...ys) * canvasElement.height;
            const maxY = Math.max(...ys) * canvasElement.height;
            const faceWidth = maxX - minX;
            const faceHeight = maxY - minY;

            // Use eye landmarks to position and rotate helmet
            // Left eye: landmark 33, Right eye: landmark 263 (MediaPipe Face Mesh)
            const leftEye = face[33];
            const rightEye = face[263];
            const eyeCenterX = ((leftEye.x + rightEye.x) / 2) * canvasElement.width;
            const eyeCenterY = ((leftEye.y + rightEye.y) / 2) * canvasElement.height;

            // Calculate rotation angle based on eye line
            const angle = Math.atan2(rightEye.y - leftEye.y, rightEye.x - leftEye.x);

            const helmetSize = faceWidth * 2.0;

            // Mirror the eye center X for correct positioning in mirrored view
            const mirroredEyeCenterX = canvasElement.width - eyeCenterX;

            if (helmetImgs[index % 5] && helmetImgs[index % 5].complete) {
              ctx.save();
              ctx.translate(mirroredEyeCenterX, eyeCenterY);
              ctx.rotate(-angle); // Negate angle for mirrored view
              ctx.drawImage(
                helmetImgs[index % 5],
                -helmetSize / 2,
                -helmetSize / 2.0,
                helmetSize,
                helmetSize
              );
              ctx.restore();
            }
          });
        }
      }
    });

    initCamera().then(() => {
      new Camera(videoElement, {
        onFrame: async () => {
          console.log("Camera frame received");
          try {
            await faceMesh.send({ image: videoElement });
            await hands.send({ image: videoElement });
          } catch (error) {
            console.error("Error processing frame:", error);
          }
        },
        width: 1280,
        height: 1080,
      }).start();
    });
